"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAlertsRoute = exports.getAlerts = void 0;
var _configSchema = require("@kbn/config-schema");
var _lib = require("../../lib");
var _constants = require("../../../../../common/signals/constants");
var _multitenancy = require("../../../../../common/multitenancy");
/*
 *    Copyright 2020 floragunn GmbH
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

const getAlerts = ({
  clusterClient,
  fetchAllFromScroll,
  logger
}) => async (context, request, response) => {
  try {
    const {
      body: {
        query,
        sort,
        index,
        scroll
      },
      headers: {
        sgtenant
      }
    } = request;
    const options = {
      index,
      scroll
    };
    if (query && !!Object.keys(query).length) {
      if (sgtenant) {
        if (!query.bool.must) {
          query.bool.must = [];
        }

        // With the new MT implementation we need to be more explicit with the
        // tenant name. Hence, the global tenant is now always SGS_GLOBAL_TENANT.
        // However, for legacy reasons "_main" needs to be used in place of
        // SGS_GLOBAL_TENANT when querying ES directly, such as in this case.
        query.bool.must.push({
          term: {
            'tenant.keyword': {
              value: sgtenant === _multitenancy.GLOBAL_TENANT_NAME ? '_main' : sgtenant
            }
          }
        });
      }
      options.body = {
        sort,
        query
      };
    }
    const firstScrollResponse = await clusterClient.asScoped(request).asCurrentUser.search(options);
    const hits = await fetchAllFromScroll({
      clusterClient,
      scroll,
      request,
      response: firstScrollResponse
    });
    return response.ok({
      body: {
        ok: true,
        resp: hits.map(h => ({
          ...h._source,
          _id: h._id,
          _index: h._index
        }))
      }
    });
  } catch (err) {
    logger.error(`getAlerts: ${err.stack}`);
    return response.customError((0, _lib.serverError)(err));
  }
};
exports.getAlerts = getAlerts;
const getAlertsRoute = ({
  router,
  clusterClient,
  fetchAllFromScroll,
  logger
}) => {
  router.post({
    path: _constants.ROUTE_PATH.ALERTS,
    validate: {
      headers: _configSchema.schema.object({
        sgtenant: _configSchema.schema.string({
          defaultValue: _constants.NO_MULTITENANCY_TENANT
        })
      }, {
        unknowns: 'allow'
      }),
      body: _configSchema.schema.object({
        index: _configSchema.schema.string({
          defaultValue: _constants.INDEX.ALERTS
        }),
        scroll: _configSchema.schema.string({
          defaultValue: _constants.ES_SCROLL_SETTINGS.KEEPALIVE
        }),
        query: _configSchema.schema.object({}, {
          unknowns: 'allow'
        }),
        sort: _configSchema.schema.arrayOf(_configSchema.schema.object({
          [_constants.DEFAULT_DATEFIELD]: _configSchema.schema.string({
            validate(value) {
              const accepted = ['desc', 'asc'];
              if (!accepted.includes(value)) {
                return `The accepted values: ${accepted.join(', ')}`;
              }
            }
          })
        }, {
          unknowns: 'allow'
        }), {
          defaultValue: {
            [_constants.DEFAULT_DATEFIELD]: 'desc'
          }
        })
      })
    }
  }, getAlerts({
    clusterClient,
    fetchAllFromScroll,
    logger
  }));
};
exports.getAlertsRoute = getAlertsRoute;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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